"use client";

import { useEffect, useState } from "react";
import { useParams, useRouter } from "next/navigation";
import Image from "next/image";
import Link from "next/link";
import { apiGet, API_ENDPOINTS } from "@/lib/api";
import { formatSalary, getSubdomain } from "@/lib/utils";
import JobMetaTags, { JobMetaPayload } from "@/app/components/JobMetaTags";

interface JobDetail extends JobMetaPayload {
  id: number;
  workplace?: string;
  salary?: string | null;
  salry_pay_by?: "range" | "fixed";
  benefits_job?: Array<{ id: number; name: string; description?: string }>;
  job_type_job?: Array<{ id: number; job_type: { name: string } }>;
  skill_job?: Array<{ id: number; name: string }>;
  company: {
    name?: string | null;
    logo?: string | null;
    website?: string | null;
    description?: string | null;
  };
  job_contact_person?: Array<{
    user: { first_name: string; last_name: string; email: string; position?: string; photo?: string };
  }>;
}

export default function JobViewPage() {
  const params = useParams();
  const router = useRouter();
  const { id, slug } = params as { id: string; slug: string };
  const [data, setData] = useState<JobDetail | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [subdomain, setSubdomain] = useState("");

  useEffect(() => {
    const sub = getSubdomain();
    setSubdomain(sub);
    fetchJob(sub);
  }, [id]);

  const fetchJob = async (sub: string) => {
    if (!id) return;
    try {
      setIsLoading(false);
      // Angular كان يستدعي API.Show + id، لذا نلحق الـ id بالمسار
      const response = await apiGet<JobDetail>(
        `${API_ENDPOINTS.JobShow}/${id}`,
        undefined,
        sub
      );
      setData(response.data);
      setIsLoading(true);
    } catch (error) {
      console.error("Error loading job", error);
      setIsLoading(true);
    }
  };

  const canonicalUrl =
    typeof window !== "undefined"
      ? window.location.href.split("?")[0]
      : "";

  if (!isLoading || !data) {
    return (
      <div className="flex min-h-screen items-center justify-center">
        <div className="text-center">
          <div className="loading-spinner"></div>
          <p className="mt-4 text-gray-600">Loading...</p>
        </div>
      </div>
    );
  }

  return (
    <>
      <JobMetaTags
        job={data}
        url={canonicalUrl}
      />
      <div className="m-page">
        <div className="row">
          <div className="col-12">
            <div className="flex items-center mt-3 mb-3 flex-wrap">
              {data.company?.logo && (
                <Image
                  src={data.company.logo}
                  alt={`${data.company.name} logo`}
                  width={120}
                  height={120}
                  className="img-comp me-3"
                  unoptimized
                />
              )}
              <div className="fs-30 fw-semibold name-header color-dark-green">
                {data.company?.name}
              </div>
            </div>

            <section className="mb-3">
              <h1 className="fs-24 fw-bold mb-3">{data.title}</h1>
            </section>

            <div className="flex justify-between flex-wrap job-info-container mb-3">
              <div className="flex flex-wrap job-info-items gap-3">
                {data.workplace && (
                  <div className="job-info-item">
                    <i className="bi bi-geo-fill color-p fw-bold fs-16 me-1"></i>
                    <span className="fs-15">{data.workplace}</span>
                  </div>
                )}
                {data.job_type_job?.length ? (
                  <div className="job-info-item">
                    <i className="bi bi-clock color-p fw-bold fs-16 me-1"></i>
                    <span className="fs-15">
                      {data.job_type_job
                        .map((t) => t.job_type.name)
                        .join(" / ")}{" "}
                      {data.hours_number ? `/ ${data.hours_number} hrs/week` : ""}
                    </span>
                  </div>
                ) : null}
                <div className="job-info-item">
                  <span className="fs-15">
                    {formatSalary(
                      data.salary || null,
                      data.min_salary || null,
                      data.max_salary || null,
                      data.currency || "USD",
                      data.rate_salary || "",
                      (data.salry_pay_by as any) || "range"
                    )}
                  </span>
                </div>
              </div>

              <div className="d-flex justify-end flex-wrap">
                <Link
                  className="btn-theme"
                  href={`/job/${slug}/${id}/apply/form`}
                >
                  Apply <i className="bi bi-chevron-right"></i>
                </Link>
              </div>
            </div>

            <div className="border-t-p"></div>

            {data.company?.description && (
              <div className="my-4">
                <h2 className="fs-20 fw-bold color-dark-green mb-3">
                  About {data.company.name}
                </h2>
                <div
                  className="mt-8 gray-6"
                  dangerouslySetInnerHTML={{ __html: data.company.description }}
                />
              </div>
            )}

            <div className="my-4">
              <div
                className="fs-16 line-h-160"
                dangerouslySetInnerHTML={{ __html: data.description }}
              />
            </div>

            {data.skill_job?.length ? (
              <div className="my-4">
                <h2 className="fs-22 fw-bold color-dark-green mb-3">
                  Required Skills
                </h2>
                <div className="d-flex flex-wrap mt-3 mb-3">
                  {data.skill_job.map((item) => (
                    <span key={item.id} className="chips">
                      {item.name}
                    </span>
                  ))}
                </div>
              </div>
            ) : null}

            {data.benefits_job?.length ? (
              <div className="my-4">
                <h2 className="fs-22 fw-bold color-dark-green mb-3">Benefits</h2>
                <div className="d-grid grid-col-3 gap-4">
                  {data.benefits_job.map((item) => (
                    <div key={item.id} className="border-gray-6 p-2 rounded-3">
                      <h4 className="fs-18 color-p mb-3">{item.name}</h4>
                      <div className="gray-6 fs-16">{item.description}</div>
                    </div>
                  ))}
                </div>
              </div>
            ) : null}

            {data.job_type_job?.length ? (
              <div className="my-4">
                <h2 className="fs-22 fw-bold color-dark-green mb-3">
                  Employment Types
                </h2>
                <div className="d-flex flex-wrap mt-3 mb-3">
                  {data.job_type_job.map((item, idx) => (
                    <span key={idx} className="chips">
                      {item.job_type.name}
                    </span>
                  ))}
                </div>
              </div>
            ) : null}

            {data.job_contact_person?.length ? (
              <div className="row mt-6">
                <div className="col-12">
                  <h2 className="fs-22 fw-bold color-dark-green mb-3">Contact Person</h2>
                </div>
                {data.job_contact_person.map((item, idx) => (
                  <div className="col-12" key={idx}>
                    <div className="d-flex justify-content-start align-items-center flex-wrap mt-3 contact-person-container">
                      <div className="me-3 mb-2 mb-md-0">
                        <Image
                          src={
                            item.user.photo || "/user-avatar-profile.svg"
                          }
                          alt="Contact person"
                          width={52}
                          height={52}
                          className="img-user"
                          unoptimized
                          onError={(e) => {
                            const target = e.target as HTMLImageElement;
                            target.src = "/user-avatar-profile.svg";
                          }}
                        />
                      </div>
                      <div className="contact-person-info">
                        <div className="fs-18 fw-bold">
                          {item.user.first_name} {item.user.last_name}
                        </div>
                        <div className="fs-16 mt-1 color-7D7D7D">
                          {item.user.email}
                        </div>
                        {item.user.position && (
                          <p className="color-7D7D7D mb-0">{item.user.position}</p>
                        )}
                      </div>
                    </div>
                  </div>
                ))}
              </div>
            ) : null}
          </div>
        </div>
      </div>
    </>
  );
}


